"""
GY-6050 (MPU6050) IMU sensor interface for orientation and motion sensing.
"""
import machine
import uasyncio as asyncio
from ..config import IMU_SCL_PIN, IMU_SDA_PIN, IMU_I2C_ID

class IMUSensor:
    def __init__(self):
        self._i2c = machine.I2C(IMU_I2C_ID, scl=machine.Pin(IMU_SCL_PIN), sda=machine.Pin(IMU_SDA_PIN))
        self._imu = None
        self._init_imu()

    def _init_imu(self):
        """Initialize MPU6050 IMU sensor."""
        try:
            from mpu6050 import MPU6050
            self._imu = MPU6050(self._i2c)
        except Exception as e:
            print(f"IMU initialization failed: {e}")

    def accel(self) -> tuple:
        """Read accelerometer data (x, y, z) in m/s²."""
        if self._imu:
            data = self._imu.get_values()
            return (data['acceleration x'], data['acceleration y'], data['acceleration z'])
        return (0.0, 0.0, 0.0)

    def gyro(self) -> tuple:
        """Read gyroscope data (x, y, z) in deg/s."""
        if self._imu:
            data = self._imu.get_values()
            return (data['gyro x'], data['gyro y'], data['gyro z'])
        return (0.0, 0.0, 0.0)

    def get_all_values(self) -> dict:
        """Get all sensor readings as a dictionary."""
        if self._imu:
            return self._imu.get_values()
        return {}

    async def monitor_orientation(self, callback=None, threshold: float = 1.0):
        """
        Monitor orientation changes and trigger callback when stage changes.
        Uses accelerometer data to detect orientation.
        """
        stage_a = (9.8 - threshold, 9.8 + threshold)
        stage_b = (-9.8 - threshold, -9.8 + threshold)
        
        current_stage = None
        detection_stage = None
        detection_start = None
        
        def in_range(x, range_tuple):
            return range_tuple[0] <= x <= range_tuple[1]
        
        while True:
            accel_x, _, _ = self.accel()
            
            # Determine current stage based on accelerometer X-axis
            if in_range(accel_x, stage_a):
                stage = 'A'
            elif in_range(accel_x, stage_b):
                stage = 'B'
            else:
                stage = None
            
            # Initialize stage if unknown
            if current_stage is None and stage in ('A', 'B'):
                current_stage = stage
            
            # Check for transition and hold duration (2 seconds)
            if stage != current_stage and stage in ('A', 'B'):
                if detection_stage is None:
                    detection_stage = stage
                    detection_start = asyncio.ticks_ms()
                elif detection_stage == stage and asyncio.ticks_diff(asyncio.ticks_ms(), detection_start) >= 2000:
                    old_stage = current_stage
                    current_stage = stage
                    detection_stage = None
                    if callback:
                        callback(old_stage, current_stage)
            else:
                detection_stage = None
            
            await asyncio.sleep_ms(100)
