import machine
import uasyncio as asyncio
from ..config import (
    VIBE_HBRIDGE_A1,
    VIBE_HBRIDGE_B1,
    DEFAULT_HB_BPM
)

class VibrationController:
    def __init__(self):
        self._a1 = machine.Pin(VIBE_HBRIDGE_A1, machine.Pin.OUT)
        self._b1 = machine.Pin(VIBE_HBRIDGE_B1, machine.Pin.OUT)
        self._active = False

    def _set_state(self, on: bool) -> None:
        """Toggle single H-bridge channel for motor control."""
        self._a1.value(on)
        self._b1.value(0)
        self._active = on

    def off(self) -> None:
        """Stop motor."""
        self._set_state(False)

    async def pulse(self, ms: int = 100) -> None:
        """Single vibration pulse of specified duration (ms)."""
        self._set_state(True)
        await asyncio.sleep_ms(ms)
        self._set_state(False)

    async def heartbeat(self, bpm: int = DEFAULT_HB_BPM) -> None:
        """
        Looping heartbeat pattern: two quick pulses per cycle.
        Call inside a task to run it continuously.
        """
        interval = 60_000 // bpm
        lub = int(0.25 * interval)
        dub = int(0.20 * interval)
        gap = interval - lub - dub - 50  # 50ms pause between pulses

        while True:
            await self.pulse(lub)
            await asyncio.sleep_ms(50)
            await self.pulse(dub)
            await asyncio.sleep_ms(gap)