"""
Non-blocking servo control with easing support for smooth motion.
Suitable for real-time environments like RP2040 Core 0.
"""
import machine, uasyncio as aio
from math import cos, pi
from ..config import SERVO_PIN, SERVO_FREQ, SERVO_MIN_US, SERVO_MAX_US

class ServoController:
    def __init__(self,
                 pin=SERVO_PIN,
                 freq=SERVO_FREQ,
                 min_us=SERVO_MIN_US,
                 max_us=SERVO_MAX_US):
        self._pwm = machine.PWM(machine.Pin(pin))
        self._pwm.freq(freq)
        self._min_us = min_us
        self._max_us = max_us
        self._current_angle = 0

    def _write_us(self, us: int):
        period = 1_000_000 // SERVO_FREQ
        duty_u16 = int(us / period * 65535)
        self._pwm.duty_u16(duty_u16)

    def angle(self, deg: float):
        """Immediately set angle (in degrees)."""
        deg = max(0, min(180, deg))
        pulse = self._min_us + (deg / 180.0) * (self._max_us - self._min_us)
        self._write_us(pulse)
        self._current_angle = deg

    async def move(self, target: float, duration: float = 1.0, easing='ease_in_out'):
        """Smooth non-blocking movement to target angle over time."""
        steps = int(duration * 50)
        start = self._current_angle
        for i in range(steps + 1):
            t = i / steps
            if easing == 'ease_in_out':
                t = 0.5 - 0.5 * cos(pi * t)
            angle = start + (target - start) * t
            self.angle(angle)
            await aio.sleep_ms(20)